implementation module windowvalidate


//	Clean Object I/O library, version 1.0.1

//	Window validation.


import StdBool, StdList, StdFunc, StdTuple, StdMisc
import StdPicture, StdSystem
import ossystem, ospicture, oswindow
import controllayout, controlvalidate, iostate, windowaccess, windowdefaccess


windowvalidateError :: String String -> .x
windowvalidateError function error
	= Error function "windowvalidate" error


/*	validateWindowId checks whether the Id of the window/dialogue has already been associated
	with open windows/dialogues. 
	If so, False is returned; otherwise True is returned and a proper Id value for the window/dialogue.
*/
validateWindowId :: !(Maybe Id) !(IOSt .l .p) -> (!Bool,Id,!IOSt .l .p)
validateWindowId Nothing ioState
	# (wDevice,ioState)	= IOStGetDevice WindowDevice ioState
	  windows			= WindowSystemStateGetWindowHandles wDevice
	  freeNrs			= windows.whsIds
	  (freeNr,freeNrs)	= HdTl freeNrs
	  windows			= {windows & whsIds=freeNrs}
	# ioState			= IOStSetDevice (WindowSystemState windows) ioState
	= (True,sysId freeNr,ioState)
validateWindowId (Just id) ioState
	# (wDevice,ioState)	= IOStGetDevice WindowDevice ioState
	  windows			= WindowSystemStateGetWindowHandles wDevice
	  (found,windows)	= hasWindowHandlesWindow (toWID id) windows
	# ioState			= IOStSetDevice (WindowSystemState windows) ioState
	| found
	= (False,undef,ioState)
	= (True, id,   ioState)


/*	Validate the given window.
*/
validateWindow :: !(WindowHandle .ls .ps) !(WindowHandles .ps) !*OSToolbox
	-> (!Index,!Point,!Size,!WindowHandle .ls .ps,!WindowHandles .ps,!*OSToolbox)

validateWindow wH=:{whMode=mode,whKind=IsDialog,whItemNrs,whItems,whAtts} windows tb
	# (wMetrics,tb)				= OSDefaultWindowMetrics tb
	  atts						= filter (not o isWindowOnlyAttribute) whAtts
	  (index,atts,windows)		= validateWindowIndex mode	atts windows
	  (pos,  atts,windows)		= validateWindowPos   mode	atts windows
	  (hasHide,atts)			= validateWindowHide  mode	atts
	  sizeAtt					= attrSize					atts
	  (hMargins,vMargins)		= attrMargins				atts wMetrics
	  spaces					= attrItemSpaces			atts wMetrics
	  defid						= getOkId					atts whItems
	  canid						= getCancelId				atts whItems
	  reqSize					= determineRequestedSize zero sizeAtt
	  (minWidth,minHeight)		= OSMinWindowSize
	  minSize					= {w=minWidth,h=minHeight}
	# (derSize,items,tb)		= layoutControls wMetrics hMargins vMargins spaces reqSize minSize zero zero whItems tb
	  (itemNrs,items)			= genWElementItemNrs whItemNrs items
	  (focusItems,items)		= getWElementKeyFocusIds (not hasHide) items
	  derSize					= determineRequestedSize derSize sizeAtt
	# okSize					= exactWindowSize wMetrics derSize Nothing Nothing IsDialog mode
	# (okPos,windows,tb)		= exactWindowPos okSize pos IsDialog mode windows tb
	= (	index
	  ,	okPos
	  ,	okSize
	  ,	{	wH	&	whItemNrs	= itemNrs
				,	whKeyFocus	= newFocusItems focusItems
				,	whItems		= items
				,	whSelect	= True
				,	whAtts		= atts
				,	whDefaultId	= defid
				,	whCancelId	= canid
				,	whSize		= okSize
		}
	  ,	windows
	  ,	tb
	  )

validateWindow wH=:{whKind=IsWindow,whItemNrs,whItems,whAtts} windows tb
	# (wMetrics,tb)				= OSDefaultWindowMetrics tb
	  atts						= filter (not o isDialogOnlyAttribute) whAtts
	  mode						= Modeless
	  (domain,atts)				= validateWindowDomain			atts
	  (maybe_hScroll,atts)		= validateWindowHScroll			atts
	  (maybe_vScroll,atts)		= validateWindowVScroll			atts
	  (look,atts)				= validateWindowLook			atts
	  (minSize,atts)			= validateMinimumSize  domain	atts
	  (reqSize,atts)			= validateSize minSize domain	atts
	  (index,atts,windows)		= validateWindowIndex mode		atts windows
	  (pos,  atts,windows)		= validateWindowPos   mode		atts windows
	  (hasHide,atts)			= validateWindowHide  mode		atts
	  (hMargins,vMargins)		= attrMargins					atts wMetrics
	  spaces					= attrItemSpaces				atts wMetrics
	  isAble					= attrSelectState				atts
	  defid						= getOkId						atts whItems
	  canid						= getCancelId					atts whItems
	# (derSize,items,tb)		= layoutControls wMetrics hMargins vMargins spaces reqSize minSize domain.corner1 zero whItems tb
	  (itemNrs,items)			= genWElementItemNrs whItemNrs items
	  (focusItems,items)		= getWElementKeyFocusIds (not hasHide) items
	  derSize					= validateDerivedSize derSize domain		// PA: windows should accept every size
	  (origin,atts)				= validateOrigin derSize domain atts
	# okSize					= exactWindowSize wMetrics derSize maybe_hScroll maybe_vScroll IsWindow mode
	# (okPos,windows,tb)		= exactWindowPos okSize pos IsWindow mode windows tb
	  (hScroll,vScroll)			= validScrollInfos wMetrics okSize maybe_hScroll maybe_vScroll
	= (	index
	  ,	okPos
	  ,	okSize
	  ,	{	wH	&	whItemNrs	= itemNrs
				,	whKeyFocus	= newFocusItems focusItems
				,	whWindowInfo= Just	{	windowDomain	= domain
										,	windowOrigin	= origin
										,	windowHScroll	= hScroll
										,	windowVScroll	= vScroll
										,	windowLook		= {lookFun=look,lookPen=defaultPen}
										,	windowClip		= {clipRgn=0,clipOk=False}
										}
				,	whItems		= items
				,	whSelect	= isAble
				,	whAtts		= atts
				,	whDefaultId	= defid
				,	whCancelId	= canid
				,	whSize		= okSize
		}
	  ,	windows
	  ,	tb
	  )
where
	validScrollInfos :: OSWindowMetrics Size !(Maybe ScrollFunction) !(Maybe ScrollFunction) -> (!Maybe ScrollInfo,!Maybe ScrollInfo)
	validScrollInfos wMetrics {w,h} maybe_hScroll maybe_vScroll
		= (scrollInfo hScrollRect maybe_hScroll,scrollInfo vScrollRect maybe_vScroll)
	where
		windowRect	= (0,0, w,h)
		hasHScroll	= isJust maybe_hScroll
		hasVScroll	= isJust maybe_vScroll
		hScrollRect	= getWindowHScrollRect wMetrics hasHScroll hasVScroll windowRect
		vScrollRect	= getWindowVScrollRect wMetrics hasHScroll hasVScroll windowRect
		
		scrollInfo :: Rect !(Maybe ScrollFunction) -> Maybe ScrollInfo
		scrollInfo _ Nothing
			= Nothing
		scrollInfo (l,t, r,b) (Just scrollFun)
			= Just {	scrollFunction	= scrollFun
				   ,	scrollItemPos	= {x=l,y=t}
  				   ,	scrollItemSize	= {w=r-l,h=b-t}
				   ,	scrollItemPtr	= OSNoWindowPtr
				   }

determineRequestedSize :: Size !(Maybe Size) -> Size
determineRequestedSize size maybe_size
	| isNothing maybe_size
	= size
	= fromJust maybe_size


/*	validateWindowIndex validates the WindowIndex attribute. 
	The return Index is the validated Index. 
	The return WindowAttribute list does not contain a WindowIndex attribute.
*/
validateWindowIndex :: !WindowMode ![WindowAttribute *(.ls,.ps)] !(WindowHandles .ps)
						-> (!Index,![WindowAttribute *(.ls,.ps)], !WindowHandles .ps)
validateWindowIndex mode atts windows=:{whsWindows}
	= (okIndex,atts`,{windows & whsWindows=modal`++modeless`})
where
	(_,indexAtt,atts`)		= Remove iswindowindex (WindowIndex 0) atts
	index					= getwindowindex indexAtt
	(modal,modeless)		= Uspan isModalWindow whsWindows
	(nrModals,modal`)		= Ulength modal
	(nrModeless,modeless`)	= Ulength modeless
	okIndex					= if (mode==Modal)
	  							 0													// Open modal windows frontmost
	  							 (SetBetween index nrModals (nrModals+nrModeless))	// Open modeless windows behind the modal windows
	
	isModalWindow :: !(WindowStateHandle .ps) -> (!Bool,!WindowStateHandle .ps)
	isModalWindow wsH
		# (mode,wsH)	= getWindowStateHandleWindowMode wsH
		= (mode==Modal,wsH)


/*	validateWindowPos validates the WindowPos attribute.
	If no WindowPos attribute is given then Nothing is returned.
	If the WindowPos is relative, it is verified that the window relates to an existing window.
	If this is not the case, then Nothing is returned.
	The resulting attribute list does not contain the WindowPos attribute anymore.
*/
validateWindowPos :: !WindowMode ![WindowAttribute *(.ls,.ps)] !(WindowHandles .ps)
			  -> (!Maybe ItemPos,![WindowAttribute *(.ls,.ps)], !WindowHandles .ps)
validateWindowPos mode atts windows
	| not hasPosAtt
	= (Nothing,atts`,windows)
	| not isRelative
	= (Just itemPos,atts`,windows)
	# (found,windows)	= hasWindowHandlesWindow (toWID relativeTo) windows
	= (if found (Just itemPos) Nothing,atts`,windows)
where
	(hasPosAtt,posAtt,atts`)	= Remove iswindowpos undef atts
	itemPos						= getwindowpos posAtt
	(isRelative,relativeTo)		= isRelativeItemPos itemPos


/*	The result ({corner1=A,corner2=B},_) of validateWindowDomain is such that A<B (point A lies to 
	the left of and above point B). If either A.x==B.x or A.y==B.y then the ViewDomain is illegal and 
	the computation is aborted. 
	The default ViewDomain is maximal and positive, i.e.:
		{corner1=zero,corner2={x=MaxSigned2ByteInt,y=MaxSigned2ByteInt}}.
	Finally, the ViewDomain must be atleast MinWindowW x MinWindowH pixels.
*/
validateWindowDomain :: ![WindowAttribute .ps] -> (!ViewDomain,![WindowAttribute .ps])
validateWindowDomain atts
	# (hasDomain,domainAtt,atts)= Remove iswindowviewdomain (WindowViewDomain (SizeToRectangle {w=MaxSigned2ByteInt,h=MaxSigned2ByteInt})) atts
	  domain					= getwindowviewdomain domainAtt
	| not hasDomain
	= (domain,atts)
	| IsEmptyRectangle domain
	= windowvalidateError "validateWindowDomain" "Window has illegal ViewDomain argument"
	= (validateViewDomain domain,atts)

validateViewDomain :: !ViewDomain -> ViewDomain
validateViewDomain domain
	= {corner1={x=l,y=t},corner2={x=max r (l+minWidth),y=max b (t+minHeight)}}
where
	(l,t, r,b)			= RectangleToRect domain
	(minWidth,minHeight)= OSMinWindowSize


/*	validateMinimumSize takes care that the minimum size attribute is at least 
	MinWindowWxMinWindowH pixels, and not larger than the given validated(!) ViewDomain.
*/
validateMinimumSize :: !ViewDomain ![WindowAttribute .ps] -> (!Size,![WindowAttribute .ps])
validateMinimumSize domain atts
	| hasMinSize
	= (minSize1, snd (Replace iswindowminimumsize (WindowMinimumSize minSize1) atts))
	with
		minSize		= getwindowminimumsize minAtt
		domainSize	= rectangleSize domain
		minSize1	= {w=SetBetween minSize.w minWidth domainSize.w,h=SetBetween minSize.h minHeight domainSize.h}
	= (dMinSize, [WindowMinimumSize dMinSize:atts])
	with
		dMinSize	= {w=minWidth,h=minHeight}
where
	(hasMinSize,minAtt)	= Select iswindowminimumsize undef atts
	(minWidth,minHeight)= OSMinWindowSize


/*	validateSize takes care that the size attribute is between the validated(!) MinimumSize and
	the validated(!) ViewDomain. The resulting size must also fit on the current screen.
*/
validateSize :: !Size !ViewDomain ![WindowAttribute .ps] -> (!Size,![WindowAttribute .ps])
validateSize minSize domain atts
	| hasSize
	= (size1,snd (Replace iswindowsize (WindowSize size1) atts))
	with
		size	= getwindowsize sizeAtt
		size1	= {w=SetBetween size.w minSize.w pictSize.w,h=SetBetween size.h minSize.h pictSize.h}
	= (pictSize,[WindowSize pictSize:atts])
where
	(hasSize,sizeAtt)	= Select iswindowsize undef atts
	domainSize			= rectangleSize domain
	wSize				= maxScrollWindowSize
	pictSize			= {w=min domainSize.w wSize.w,h=min domainSize.h wSize.h}


/*	validateDerivedSize takes care that the derived size of layoutControls does not exceed
	the validated(!) ViewDomain.
*/
validateDerivedSize :: !Size !ViewDomain -> Size
validateDerivedSize {w,h} domain
	= {w=min w domainSize.w,h=min h domainSize.h}
where
	domainSize	= rectangleSize domain


/*	validateOrigin takes care that the WindowOrigin attribute is a point in the rectangle
	formed by the left top of the (validated!) ViewDomain, and the width and height of the 
	(validated!) derived size.
*/
validateOrigin :: !Size !ViewDomain ![WindowAttribute .ps] -> (!Point,![WindowAttribute .ps])
validateOrigin {w,h} domain=:{corner1={x=l,y=t},corner2={x=r,y=b}} atts
	# (_,domainAtt,atts)	= Remove iswindoworigin (WindowOrigin domain.corner1) atts
	  origin				= getwindoworigin domainAtt
	= ({x=SetBetween origin.x l (r-w),y=SetBetween origin.y t (b-h)},atts)


/*	validateWindow(H/V)Scroll removes the Window(H/V)Scroll attribute from the attribute list. 
*/
validateWindowHScroll :: ![WindowAttribute .ps] -> (!Maybe ScrollFunction,![WindowAttribute .ps])
validateWindowHScroll atts
	# (found,scrollAtt,atts)	= Remove iswindowhscroll undef atts
	| found
	= (Just (getwindowhscrollfunction scrollAtt),atts)
	= (Nothing,atts)

validateWindowVScroll :: ![WindowAttribute .ps] -> (!Maybe ScrollFunction,![WindowAttribute .ps])
validateWindowVScroll atts
	# (found,scrollAtt,atts)	= Remove iswindowvscroll undef atts
	| found
	= (Just (getwindowvscrollfunction scrollAtt),atts)
	= (Nothing,atts)


/*	validateWindowHide takes care that Modal windows are not opened invisibly.
*/
validateWindowHide :: !WindowMode ![WindowAttribute .ps] -> (!Bool,![WindowAttribute .ps])
validateWindowHide Modal atts
	= (False,thd3 (Remove iswindowhide undef atts))
validateWindowHide _ atts
	= (Contains iswindowhide atts,atts)


/*	validateWindowLook takes care that the optional WindowLook attribute is removed from the attribute list.
	If no attribute was present, then a default look is provided that paints the window White.
*/
validateWindowLook :: ![WindowAttribute .ps] -> (!Look,![WindowAttribute .ps])
validateWindowLook atts
	# (hasLook,lookAtt,atts)	= Remove iswindowlook undef atts
	| hasLook
	= (validateLook (getwindowlook lookAtt),atts)
	= (defaultLook,atts)
where
	defaultLook :: SelectState !UpdateState -> [DrawFunction]
	defaultLook _ {newFrame}
//		= [drawPicture [setPenColour White,fill newFrame]]
		= [setPenColour White,fill newFrame]


//	Retrieve size, margins, item spaces, and select state from the attribute list.

attrSize :: ![WindowAttribute .ps] -> Maybe Size
attrSize atts
	| hasSize
	= Just (getwindowsize att)
	= Nothing
where
	(hasSize,att)	= Select iswindowsize undef atts

attrMargins :: ![WindowAttribute .ps] !OSWindowMetrics -> ((Int,Int),(Int,Int))
attrMargins atts {osmHorMargin,osmVerMargin}
	= (hMargins,vMargins)
where
	hMargins = getwindowhmargin (snd (Select iswindowhmargin (WindowHMargin osmHorMargin osmHorMargin) atts))
	vMargins = getwindowvmargin (snd (Select iswindowvmargin (WindowVMargin osmVerMargin osmVerMargin) atts))

attrItemSpaces :: ![WindowAttribute .ps] !OSWindowMetrics -> (Int,Int)
attrItemSpaces atts {osmHorItemSpace,osmVerItemSpace}
	= getwindowitemspace (snd (Select iswindowitemspace (WindowItemSpace osmHorItemSpace osmVerItemSpace) atts))

attrSelectState :: ![WindowAttribute .ps] -> Bool
attrSelectState atts
	= enabled (getwindowselectstate (snd (Select iswindowselectstate (WindowSelectState Able) atts)))



/*	get(Ok/Cancel)Id select the Id of the Window(Ok/Cancel) attribute, and checks
	whether this Id corresponds with a (Custom)ButtonControl.
*/
getOkId :: ![WindowAttribute (.ls,.ps)] ![WElementHandle .ls .ps] -> Maybe Id
getOkId atts itemHs
	| hasid
	= Just (getwindowok idAtt)
	= Nothing
where
	(hasid,idAtt)	= Select iswindowok undef atts

getCancelId :: ![WindowAttribute (.ls,.ps)] ![WElementHandle .ls .ps] -> Maybe Id
getCancelId atts items
	| hasid
	= Just (getwindowcancel idAtt)
	= Nothing
where
	(hasid,idAtt)	= Select iswindowcancel undef atts


/*	exactWindowSize determines the exact size of a window.
	The size is extended to fit in sliderbars if requested (argument 3 and 4).
*/
exactWindowSize :: OSWindowMetrics !Size (Maybe ScrollFunction) (Maybe ScrollFunction) !WindowKind !WindowMode -> Size
exactWindowSize wMetrics wSize maybe_hScroll maybe_vScroll wKind wMode
	| wKind==IsDialog && wMode==Modal
	= wSize
	| hasHScroll && hasVScroll
	= {w=w`,h=h`}
	| hasHScroll
	= {wSize & h=h`}
	| hasVScroll
	= {wSize & w=w`}
	= wSize
where
	hasHScroll	= isJust maybe_hScroll
	hasVScroll	= isJust maybe_vScroll
	w`			= wSize.w+wMetrics.osmVSliderWidth -1
	h`			= wSize.h+wMetrics.osmHSliderHeight-1


/*	exactWindowPos determines the exact position of a window.
	The size argument must be the exact size as calculated by exactWindowSize of the window.
	The ItemPos argument must be the validated(!) ItemPos attribute of the window.
*/
exactWindowPos :: !Size !(Maybe ItemPos) !WindowKind !WindowMode !(WindowHandles .ps) !*OSToolbox
													  -> (!Point, !WindowHandles .ps, !*OSToolbox)
exactWindowPos exactSize maybePos wKind wMode windows tb
	| wKind==IsDialog && wMode==Modal
	= (pos,windows,tb1)
	with
		((sl,st,sr,sb),tb1)	= OSscreenrect tb
		l					= sl + (sr-sl-exactSize.w)/2
		t					= st + (sb-st-exactSize.h)/3
		pos					= {x=SetBetween l sl sr,y=SetBetween t st sb}
	| isNothing maybePos
	= (zero,windows,tb)
	# itemPos				= fromJust maybePos
	# (pos,windows,tb)		= getItemPosPosition exactSize itemPos windows tb
	# (pos,tb)				= setWindowInsideScreen pos exactSize tb
	= (pos,windows,tb)
where
/*	getItemPosPosition calculates the exact position of the given window. 
	getItemPosPosition does not check whether this position will place the window off screen.
*/
	getItemPosPosition :: !Size !ItemPos !(WindowHandles .ps) !*OSToolbox -> (!Point,!WindowHandles .ps,!*OSToolbox)
	getItemPosPosition size itemPos windows=:{whsWindows=wsHs} tb
		| isRelative
		# (before,[wsH=:{wshIds={wPtr}}:after])
										= Uspan (identifyWindow (toWID relativeTo)) wsHs
		  windows						= {windows & whsWindows=before++[wsH:after]}
		# ((relativeX,relativeY),tb)	= OSgetWindowPos  wPtr tb
		# ((relativeW,relativeH),tb)	= OSgetWindowSize wPtr tb
		  (exactW,exactH)				= (size.w,size.h)
		  pos							= case itemPos of
					  						(LeftOf  _,{vx,vy})	-> {x=relativeX+vx-exactW,   y=relativeY+vy}
					  						(RightTo _,{vx,vy})	-> {x=relativeX+vx+relativeW,y=relativeY+vy}
			  								(Above   _,{vx,vy})	-> {x=relativeX+vx,          y=relativeY+vy-exactH}
		  									(Below   _,{vx,vy})	-> {x=relativeX+vx,          y=relativeY+vy+relativeH}
		= (pos,windows,tb)
	where
		(isRelative,relativeTo)	= isRelativeItemPos itemPos
		
		identifyWindow :: !WID !(WindowStateHandle .ps) -> (!Bool,!WindowStateHandle .ps)
		identifyWindow wid wsH
			# (ids,wsH)			= getWindowStateHandleWIDS wsH
			= (identifyWIDS wid ids,wsH)
	getItemPosPosition size itemPos windows tb
		| isAbsolute
		= (pos,windows,tb)
	where
		(isAbsolute,pos)	= isAbsoluteItemPos itemPos
	getItemPosPosition size itemPos windows tb
		| isCornerItemPos itemPos
		# ((l,t,r,b),tb)	= OSscreenrect tb
		  (exactW, exactH)	= (size.w,size.h)
		  pos				= case itemPos of
		  						(LeftTop,    {vx,vy})	-> {x=l+vx,       y=t+vy}
		  						(RightTop,   {vx,vy})	-> {x=r+vx-exactW,y=t+vy}
		  						(LeftBottom, {vx,vy})	-> {x=l+vx,       y=b+vy-exactH}
		  						(RightBottom,{vx,vy})	-> {x=r+vx-exactW,y=b+vy-exactH}
		= (pos,windows,tb)
	getItemPosPosition size itemPos windows tb
		= (zero,windows,tb)
	
/*	setWindowInsideScreen makes sure that a window at the given position and given size will be on screen.
*/
	setWindowInsideScreen :: !Point !Size !*OSToolbox -> (!Point,!*OSToolbox)
	setWindowInsideScreen pos=:{x,y} size=:{w,h} tb
		# ((l,t,r,b),tb)	= OSscreenrect tb
		  (screenW,screenH)	= (r-l,b-t)
		  (x`,y`)			= (SetBetween x l (r-w),SetBetween y t (b-h))
		  pos				= if (w<=screenW && h<=screenH)	{x=x`,y=y`}			// window fits entirely on screen
		  					 (if (w<=screenW)				{x=x`,y=0 }			// window is to high
		  					 (if (h<=screenH)				{x=0, y=y`}			// window is to wide
		  					 (								zero)))				// window doesn't fit anyway
		= (pos,tb)


//	Predicates on ItemPos:
isRelativeItemPos :: !ItemPos -> (!Bool,Id)
isRelativeItemPos (LeftOf  id,_)	= (True, id)
isRelativeItemPos (RightTo id,_)	= (True, id)
isRelativeItemPos (Above   id,_)	= (True, id)
isRelativeItemPos (Below   id,_)	= (True, id)
isRelativeItemPos _					= (False,undef)

isAbsoluteItemPos :: !ItemPos -> (!Bool,Point)
isAbsoluteItemPos (Fix point,_)		= (True, point)
isAbsoluteItemPos _					= (False,undef)

isCornerItemPos :: !ItemPos -> Bool
isCornerItemPos (LeftTop,_)			= True
isCornerItemPos (RightTop,_)		= True
isCornerItemPos (LeftBottom,_)		= True
isCornerItemPos (RightBottom,_)		= True
isCornerItemPos _					= False
